"""Task assignment model linking tasks to employees."""

from datetime import date
from enum import Enum
from typing import Optional
from pydantic import BaseModel, Field, validator


class AssignmentStatus(str, Enum):
    PENDING = "Pending"
    IN_PROGRESS = "In Progress"
    COMPLETED = "Completed"
    DEFERRED = "Deferred"


class Assignment(BaseModel):
    id: Optional[int] = Field(default=None)
    task_id: int
    employee_id: int
    assigned_on: date
    assigned_by: int  # user id who assigned
    status: AssignmentStatus = AssignmentStatus.PENDING
    percent_complete: int = Field(default=0, ge=0, le=100)
    remarks: Optional[str] = None
    actual_completion_date: Optional[date] = None

    class Config:
        orm_mode = True

    @validator("actual_completion_date")
    def validate_completion_date(cls, v: Optional[date], values):
        # Completion date should not be before assignment date
        if v is not None and "assigned_on" in values and v < values["assigned_on"]:
            raise ValueError("Completion date cannot be before assignment date.")
        return v